
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 25 10:17:10 2007
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version CRYS_ATP_TST_FLOW.c#1:csrc:1
   *  \author Annag
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/

#include "CRYS.h"

#include "CRYS_ATP_TST_Defines.h"
#include "CRYS_ATP_TST_UTIL_Funcs.h"
#include "CRYS_ATP_TST_UTIL_OTF_Funcs.h"
#include "CRYS_ATP_TST_AESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_AESFLOW_Acceptance_test_vectors.h"
#include "CRYS_ATP_TST_DESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_DESFLOW_Acceptance_test_vectors.h"
#include "CRYS_ATP_TST_HASHFLOW_Acceptance.h"
#include "CRYS_ATP_TST_HASHFLOW_Acceptance_test_vectors.h"
#include "CRYS_ATP_TST_RC4FLOW_Acceptance.h"
#include "CRYS_ATP_TST_RC4FLOW_Acceptance_test_vectors.h"
#include "CRYS_ATP_TST_HMACFLOW_Acceptance.h"
#include "CRYS_ATP_TST_HMACFLOW_Acceptance_test_vectors.h"

#include "CRYS_ATP_TST_FLOW_Defines.h"


#include "gen.h"


#include "CRYS_INIT_TST.h"
#include "CRYS_FLOW.h"
#include "CRYS_FLOW_error.h"

#include "DX_VOS_Mem.h"

#if TST_USE_FLOW_DATA


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern DxUint8_t 	                 TST_Input_Data[TST_FLOW_DATA_VEC_SIZE_IN_BYTS];

extern DxUint64_t	                 TST_FLOW_InputData[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

extern DxUint64_t	                 TST_FLOW_OutputData[TST_FLOW_NUMBER_OF_SUPPORTED_ENGINES][TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];


/* input data size */
extern DxUint32_t                    TST_FLOW_InputDataSize;


/* AES engine data */
extern CRYS_FLOW_AESEngineData_t     TST_FLOWAESEngineData;

/* DES engine data */
extern CRYS_FLOW_DESEngineData_t     TST_FLOWDESEngineData;

/* HASH engine data */
extern CRYS_FLOW_HASHEngine_Data_t   TST_FLOWHASHEngineData;

/* common data for one test vector */
extern TST_FLOW_TestCommonDataStruct TST_TestData;

/* output data of HASH/AES_MAC result */
extern CRYS_AES_MAX_MAC_RESULT_t     TST_FLOW_AESMAC_OutputData;
/* output data of HASH/AES_MAC result */
extern DxUint32_t                    TST_FLOW_Hash_OutputData[CRYS_HASH_RESULT_SIZE_IN_WORDS];

/* flow params info */
CRYS_FLOW_Params_t                   TST_AESFLOWParamsInfo;

CRYS_FLOW_Params_t                   TST_DESFLOWParamsInfo;

CRYS_FLOW_Params_t                   TST_HASHFLOWParamsInfo;

#if FLOW_RC4_Support

CRYS_FLOW_Params_t                   TST_RC4FLOWParamsInfo;

#endif



/************* Private function prototype ***********************/

 
/************************ Public Functions ******************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_PARALLEL_Acceptance
 *  
 * Inputs:
 *  TST_index - test index,
 *  TST_MasterTestName - matser test name,
 *  flowType - flow type
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for FLOW API's running in parallel
 
 * Algorithm:     
 *	1. Init AES flow
 *  2. Send interrupt that AES input/output buffers are ready
 *  3. Start DES flow
 *  4. Send interrupt that DES input/output buffers are ready
 *  5. Start HASH flow
 *  6. Start RC4 flow
 *  7. Continue AES flow
 *  8. Continue RC4 flow
 *  9. Finish AES flow and compare results 
 * 10. Execute some AES operation and compare with expected result
 * 11: Finish RC4 flow and compare results
 * 12: Continue DES flow 
 * 13: Terminate AES flow
 * 14: Run cyclic AES flow
 * 15: Run HMAC flow
 * 16: Continue HASH flow
 * 17: Finish HASH flow and compare results
 * 18: Finish DES flow and compare results
 * 19: Terminate HASH flow
 * 20: Terminate DES flow
 * 21: Terminate RC4 flow
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_PARALLEL_Acceptance ( int              TST_index, 
                                                    char             *TST_MasterTestName,
                                                    CRYS_FLOW_Type_t flowType ) 
{

    /********* TST Variables Definitions ************/
    CRYSError_t             	TST_TestStatus = TESTPASS;
 
 	/* flow id */
    DxUint32_t                   TST_AESFlowId  = 0;
    DxUint32_t                   TST_DESFlowId  = 0;
    DxUint32_t                   TST_HASHFlowId = 0;
    
    #if FLOW_RC4_Support    
    DxUint32_t                   TST_RC4FlowId  = 0;
    #endif
    
    DxUint32_t                   i = 0;
    
    /* input block size */
	DxUint16_t                   blockSize = 0;
    
   
    
  	/************************************************/
    /*********** Start of test **********************/
    /************************************************/
    
    /* prepare input buffer: */
	DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
    
    /* Init AES flow */
    /***********************/
    TST_TestStatus = CRYS_ATP_TST_InitAESFlow(TST_index, flowType, TST_MasterTestName, &TST_AESFlowId, &TST_AESFLOWParamsInfo, CRYS_AES_Encrypt);
    
    blockSize = TST_FLOW_InputDataSize/TST_AESFLOWParamsInfo.inputBuffersArraySize;
         	
    
    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
            
    /* Send interrupt that AES input/output buffers are ready */
    /******************************************************/
    
    /* Send interrupt that input buffers are ready */
    CRYS_ATP_TST_FLOW_SendInterrupt(TST_AESFLOWParamsInfo.inputInterruptId);
    
    /* Send interrupt that output buffers are ready */
    CRYS_ATP_TST_FLOW_SendInterrupt(TST_AESFLOWParamsInfo.outputInterruptId);
    
    
   
    /* Start DES flow */
    /******************/
    TST_TestStatus = CRYS_ATP_TST_InitDESFlow(TST_index, flowType, TST_MasterTestName, 
                                              &TST_DESFlowId, &TST_DESFLOWParamsInfo, CRYS_DES_Encrypt);
                                              
    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif                                                
                                              
    
    /* Send interrupt that DES input/output buffers are ready */
    /******************************************************/
    
    /* Send interrupt that input buffers are ready */
    CRYS_ATP_TST_FLOW_SendInterrupt(TST_DESFLOWParamsInfo.inputInterruptId);
    
    /* Send interrupt that output buffers are ready */
    CRYS_ATP_TST_FLOW_SendInterrupt(TST_DESFLOWParamsInfo.outputInterruptId);                                                      
    
    
    
    /* start HASH flow */
    /******************/
    TST_TestStatus = CRYS_ATP_TST_InitHASHFlow (TST_index, flowType, TST_MasterTestName, 
                                                &TST_HASHFlowId, &TST_HASHFLOWParamsInfo);
                                                
    #if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
    #endif  
                                                                                            
    /* Send interrupt that HASH input buffer is ready */
    /******************************************************/
    
    /* Send interrupt that input buffers are ready */
    CRYS_ATP_TST_FLOW_SendInterrupt(TST_HASHFLOWParamsInfo.inputInterruptId);
                                        
    /* start RC4 flow */
	/******************/
    #if FLOW_RC4_Support
	
	TST_TestStatus = CRYS_ATP_TST_InitRC4Flow(TST_index, flowType, TST_MasterTestName, 
	                                          &TST_RC4FlowId, &TST_RC4FLOWParamsInfo);
	                                          
    #endif	                                          
    

    /* continue AES flow */
    /**************************/
    for (i = 0; i < TST_AESFLOWParamsInfo.inputBuffersArraySize - 1 - 1; ++i)
        /* Send interrupt that input buffers are ready */
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_AESFLOWParamsInfo.inputInterruptId);
        
    for (i = 0; i < TST_AESFLOWParamsInfo.outputBuffersArraySize - 1; ++i)
        /* Send interrupt that output buffers are ready */
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_AESFLOWParamsInfo.outputInterruptId);
        
    
    #if FLOW_RC4_Support
    
    /* continue RC4 flow */
    /**************************/
    for (i = 0; i < TST_RC4FLOWParamsInfo.inputBuffersArraySize; ++i)
        /* Send interrupt that input buffers are ready */
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_RC4FLOWParamsInfo.inputInterruptId);
        
    for (i = 0; i < TST_RC4FLOWParamsInfo.outputBuffersArraySize; ++i)
        /* Send interrupt that output buffers are ready */
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_RC4FLOWParamsInfo.outputInterruptId);    
    #endif        
    
   		   
            
    /* Finish AES flow and compare results */
    /***************************************/
    TST_TestStatus = CRYS_ATP_TST_FinishAESFlow(TST_index, TST_AESFlowId, TST_MasterTestName, blockSize, &TST_AESFLOWParamsInfo, CRYS_AES_Encrypt);
    
	#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
    	return TESTFAIL;
    #endif 
    
   
    /* execute some AES operation and compare with expected result */
    /***************************************************************/
    TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckAESOperation();

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"AES operation",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif 


    #if FLOW_RC4_Support    
    /* Finish RC4 flow and compare results */
    /***************************************/
    TST_TestStatus = CRYS_ATP_TST_FinishRC4Flow(TST_index, TST_RC4FlowId, TST_MasterTestName, 0, &TST_RC4FLOWParamsInfo);
    
	#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
    	return TESTFAIL;
    #endif  
    #endif    

        
    /* continue DES flow */
    /**************************/
    /* Send interrupt that input buffers are ready */
    for (i = 0; i < TST_DESFLOWParamsInfo.inputBuffersArraySize - 1; ++i)
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_DESFLOWParamsInfo.inputInterruptId);
    
    /* Send interrupt that output buffers are ready */
    for (i = 0; i < TST_DESFLOWParamsInfo.outputBuffersArraySize - 1; ++i)        
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_DESFLOWParamsInfo.outputInterruptId);
        
    /* Terminate AES flow */
	/*******************/
	TST_ATP_Error = CRYS_FLOW_Terminate(TST_AESFlowId);
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"AES flow terminate is done\n"));                             
		
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

   	#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
    	return TESTFAIL;
    #endif      
        
    /* run cyclic AES flow */
    /***********************/
    CRYS_ATP_TST_FLOW_AESSTATIC_ONETEST_Cyclic_Acceptance(TST_index, TST_MasterTestName);
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"AES cyclic flow",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
    if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif

    /* Run HMAC flow */
    /*****************/
    TST_TestStatus = CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeStatic);
		

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

    /* Continue HASH flow */
    /**********************/
    for (i = 0; i < TST_HASHFLOWParamsInfo.inputBuffersArraySize - 1; ++i)
        /* Send interrupt that input buffers are ready */
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_HASHFLOWParamsInfo.inputInterruptId);
        
    
    /* Finish HASH flow and compare results */
    /********************************************************************/
    TST_TestStatus = CRYS_ATP_TST_FinishHASHFlow(TST_index, TST_HASHFlowId, TST_MasterTestName, 0, &TST_HASHFLOWParamsInfo);
    
    
    /* Finish DES flow and compare results */
    /********************************************************************/
    TST_TestStatus = CRYS_ATP_TST_FinishDESFlow(TST_index, TST_DESFlowId, TST_MasterTestName, 0, &TST_DESFLOWParamsInfo, CRYS_DES_Encrypt);
    
    
    /* Terminate HASH flow */
	/***********************/
    TST_ATP_Error = CRYS_FLOW_Terminate(TST_HASHFlowId);
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"HASH flow terminate is done\n"));                             
		
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

   	#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
    	return TESTFAIL;
    #endif  
    
    /* Terminate DES flow */
	/***********************/
    TST_ATP_Error = CRYS_FLOW_Terminate(TST_DESFlowId);
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"DES flow terminate is done\n"));                             
		
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

   	#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
    	return TESTFAIL;
    #endif  
    
    #if FLOW_RC4_Support    
    /* Terminate RC4 flow */
	/***********************/
    TST_ATP_Error = CRYS_FLOW_Terminate(TST_RC4FlowId);
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"RC4 flow terminate is done\n"));                             
		
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_RC4FLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

   	#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
    	return TESTFAIL;
    #endif  
    #endif    
	                           
  	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_PARALLEL_Acceptance*/ 


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_PARALLEL_DYNAMIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for FLOW API's
 
 * Algorithm:     
 *	This function goes through several test vectors and runs 
 *  CRYS_ATP_TST_FLOW_PARALLEL_Acceptance test for each vector.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_PARALLEL_DYNAMIC_Acceptance ( void ) 
{

    /********* TST Variables Definitions ************/
    int                         TST_index;
    CRYSError_t             	TST_TestStatus = TESTPASS;
    char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_PARALLEL_DYNAMIC_Acceptance"; 
	
    
 
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_PARALLEL_DYNAMIC_Acceptance Tests ********************* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));
    
    /* go through several test vectors and runs CRYS_ATP_TST_FLOW_PARALLEL_Acceptance test */
    /* for each vector                                                                     */
    for (TST_index =0 ; TST_index < TST_FLOW_COMMON_NUMBER_OF_VECTORS ; TST_index++) /*To run on all test vectors*/
	{
        TST_TestStatus = CRYS_ATP_TST_FLOW_PARALLEL_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeDynamic);
        
        #if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
        #endif  

	}

    return TST_TestStatus;
}

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_PARALLEL_STATIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for FLOW API's
 
 * Algorithm:     
 *	This function goes through several test vectors and runs 
 *  CRYS_ATP_TST_FLOW_PARALLEL_Acceptance test for each vector.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_PARALLEL_STATIC_Acceptance ( void ) 
{

    /********* TST Variables Definitions ************/
    int                         TST_index;
    CRYSError_t             	TST_TestStatus = TESTPASS;
    char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_PARALLEL_STATIC_Acceptance"; 
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_PARALLEL_STATIC_Acceptance Tests ********************* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));
    
    /* go through several test vectors and runs CRYS_ATP_TST_FLOW_PARALLEL_Acceptance test */
    /* for each vector                                                                     */
    for (TST_index =0 ; TST_index < TST_FLOW_COMMON_NUMBER_OF_VECTORS ; TST_index++) /*To run on all test vectors*/
	{
        TST_TestStatus = CRYS_ATP_TST_FLOW_PARALLEL_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeStatic);
	}
    
 
  	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_PARALLEL_STATIC_Acceptance*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_SEQUENTIALLY_Acceptance
 *  
 * Inputs:
 *  TST_index - test index,
 *  TST_MasterTestName - matser test name,
 *  flowType - flow type
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for FLOW API's running in sequence
 
 * Algorithm:     
 * 1. Run HASH flow	
 * 2. Run HASH flow
 * 3. Run AES flow
 * 4. Run DES flow
 * 5. Run RC4 flow
 * 6. Run HMAC flow
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_SEQUENTIALLY_Acceptance ( int              TST_index, 
                                                        char             *TST_MasterTestName,
                                                        CRYS_FLOW_Type_t flowType ) 
{

    /********* TST Variables Definitions ************/
    CRYSError_t             	TST_TestStatus = TESTPASS;
	
   
  	/************************************************/
    /*********** Start of test **********************/
    /************************************************/
    
    /* Run HASH flow */
    /*****************/
    TST_TestStatus = CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance(TST_index, TST_MasterTestName, flowType);
        
    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
    
    /* Run HASH flow */
    /*****************/
    TST_TestStatus = CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance(TST_index, TST_MasterTestName, flowType);
    
    
    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
    
    /* Run AES flow */
    /*****************/
    TST_TestStatus = CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance(TST_index, TST_MasterTestName, flowType);
    
    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
            
    /* Run DES flow */
    /*****************/
    TST_TestStatus = CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance(TST_index, TST_MasterTestName, flowType);
    
    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
    
    #if FLOW_RC4_Support    
    /* Run RC4 flow */
    /*****************/
    TST_TestStatus = CRYS_ATP_TST_FLOW_RC4_ONETEST_Acceptance(TST_index, TST_MasterTestName, flowType);
    
    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
    #endif    
    
    /* Run HMAC flow */
    /*****************/
    TST_TestStatus = CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance(TST_index, TST_MasterTestName, flowType);
    
    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
    
    return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_SEQUENTIALLY_Acceptance*/ 


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_SEQUENTIALLY_DYNAMIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for FLOW API's
 
 * Algorithm:     
 *	This function goes through several test vectors and runs 
 *  CRYS_ATP_TST_FLOW_SEQUENTIALLY_Acceptance test for each vector.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_SEQUENTIALLY_DYNAMIC_Acceptance () 
{

    /********* TST Variables Definitions ************/
    int                         TST_index;
    CRYSError_t             	TST_TestStatus = TESTPASS;
    char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_SEQUENTIALLY_DYNAMIC_Acceptance"; 
	
   	
  
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_SEQUENTIALLY_DYNAMIC_Acceptance Tests ********************* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));
    
    for (TST_index =0 ; TST_index < TST_FLOW_COMMON_NUMBER_OF_VECTORS ; TST_index++) /*To run on all test vectors*/
    {
        TST_TestStatus = CRYS_ATP_TST_FLOW_SEQUENTIALLY_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeDynamic);
    }
  	
    return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_SEQUENTIALLY_DYNAMIC_Acceptance*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_SEQUENTIALLY_STATIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for FLOW API's
 
 * Algorithm:     
 *	This function goes through several test vectors and runs 
 *  CRYS_ATP_TST_FLOW_SEQUENTIALLY_Acceptance test for each vector.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_SEQUENTIALLY_STATIC_Acceptance ( void ) 
{

    /********* TST Variables Definitions ************/
    int                         TST_index;
    CRYSError_t             	TST_TestStatus = TESTPASS;
    char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_SEQUENTIALLY_STATIC_Acceptance"; 
	
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_SEQUENTIALLY_STATIC_Acceptance Tests ********************* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));
    
    for (TST_index =0 ; TST_index < TST_FLOW_COMMON_NUMBER_OF_VECTORS ; TST_index++) /*To run on all test vectors*/
    {
        TST_TestStatus = CRYS_ATP_TST_FLOW_SEQUENTIALLY_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeStatic);
    }
  	
    return TST_TestStatus;    
    
}/* End of CRYS_ATP_TST_FLOW_SEQUENTIALLY_STATIC_Acceptance*/ 


/***************************
 * Function Name:
 * 	CRYS_ATP_TST_FLOW_STATIC_Acceptance
 *
 * Inputs:
 * 	none
 *
 * Outputs:
 * 	CRYSError_t - Function error return
 *
 * Description:
 * This function runs static FLOW Acceptance tests for all engines.
 *
 **************************/
CRYSError_t CRYS_ATP_TST_FLOW_STATIC_Acceptance(void)
{
    CRYSError_t TST_ATP_FLOW_Error = TESTPASS;

    TST_ATP_Error = CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    #if FLOW_RC4_Support    
    TST_ATP_FLOW_Error = CRYS_ATP_TST_FLOW_RC4STATIC_Acceptance();
    
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    #endif    
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance();
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
        
   return TST_ATP_FLOW_Error;
}

/***************************
 * Function Name:
 * 	CRYS_ATP_TST_FLOW_STATIC_Acceptance_WithAdditionalEngines
 *
 * Inputs:
 * 	none
 *
 * Outputs:
 * 	CRYSError_t - Function error return
 *
 * Description:
 * This function runs static FLOW Acceptance tests for all engines
 * with additional engines at the middle of the flow.
 *
 **************************/
CRYSError_t CRYS_ATP_TST_FLOW_STATIC_Acceptance_WithAdditionalEngines()
{
    CRYSError_t TST_ATP_FLOW_Error = TESTPASS;

    TST_ATP_Error = CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance_WithAdditionalEngines();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance_WithAdditionalEngines();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance_WithAdditionalEngines();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    #if FLOW_RC4_Support    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_RC4STATIC_Acceptance_WithAdditionalEngines();
    
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    #endif    
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance_WithAdditionalEngines();
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
        
   return TST_ATP_FLOW_Error;
}

/***************************
 * Function Name:
 * 	CRYS_ATP_TST_FLOW_STATIC_Cyclic_Acceptance
 *
 * Inputs:
 * 	none
 *
 * Outputs:
 * 	CRYSError_t - Function error return
 *
 * Description:
 * This function runs static cyclic FLOW Acceptance tests for all engines.
 *
 **************************/
CRYSError_t CRYS_ATP_TST_FLOW_STATIC_Cyclic_Acceptance()
{
    CRYSError_t TST_ATP_FLOW_Error = TESTPASS;

    TST_ATP_Error = CRYS_ATP_TST_FLOW_AESSTATIC_Cyclic_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_DESSTATIC_Cyclic_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HASHSTATIC_Cyclic_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    #if FLOW_RC4_Support    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_RC4STATIC_Cyclic_Acceptance();
    
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HMACSTATIC_Cyclic_Acceptance();
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
        
   return TST_ATP_FLOW_Error;
}

/***************************
 * Function Name:
 * 	CRYS_ATP_TST_FLOW_DYNAMIC_Acceptance
 *
 * Inputs:
 * 	none
 *
 * Outputs:
 * 	CRYSError_t - Function error return
 *
 * Description:
 * This function runs dynamic FLOW Acceptance tests for all engines.
 *
 **************************/
CRYSError_t CRYS_ATP_TST_FLOW_DYNAMIC_Acceptance(void)
{
    CRYSError_t TST_ATP_FLOW_Error = TESTPASS;

    TST_ATP_Error = CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    #if FLOW_RC4_Support
    TST_ATP_Error = CRYS_ATP_TST_FLOW_RC4DYNAMIC_Acceptance();
    #endif
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance();
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
        
   return TST_ATP_FLOW_Error;
}

/***************************
 * Function Name:
 * 	CRYS_ATP_TST_FLOW_DYNAMIC_Acceptance_WithAdditionalEngines
 *
 * Inputs:
 * 	none
 *
 * Outputs:
 * 	CRYSError_t - Function error return
 *
 * Description:
 * This function runs dynamic FLOW Acceptance tests for all engines
 * with additional engines at the middle of the flow.
 *
 **************************/
CRYSError_t CRYS_ATP_TST_FLOW_DYNAMIC_Acceptance_WithAdditionalEngines()
{
    CRYSError_t TST_ATP_FLOW_Error = TESTPASS;

    TST_ATP_Error = CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance_WithAdditionalEngines();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance_WithAdditionalEngines();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance_WithAdditionalEngines();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    #if FLOW_RC4_Support
    TST_ATP_Error = CRYS_ATP_TST_FLOW_RC4DYNAMIC_Acceptance_WithAdditionalEngines();
    #endif
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance_WithAdditionalEngines();
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
        
   return TST_ATP_FLOW_Error;
}

/***************************
 * Function Name:
 * 	CRYS_ATP_TST_FLOW_DYNAMIC_AddBuffers_Acceptance
 *
 * Inputs:
 * 	none
 *
 * Outputs:
 * 	CRYSError_t - Function error return
 *
 * Description:
 * This function runs dynamic FLOW Acceptance tests for all engines
 * with add buffers functionality.
 *
 **************************/
CRYSError_t CRYS_ATP_TST_FLOW_DYNAMIC_AddBuffers_Acceptance()
{
    CRYSError_t TST_ATP_FLOW_Error = TESTPASS;

    TST_ATP_Error = CRYS_ATP_TST_FLOW_AESDYNAMIC_AddBuffers_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_DESDYNAMIC_AddBuffers_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HASHDYNAMIC_AddBuffers_Acceptance();
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    #if FLOW_RC4_Support
    TST_ATP_Error = CRYS_ATP_TST_FLOW_RC4DYNAMIC_AddBuffers_Acceptance();
    #endif
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
    TST_ATP_Error = CRYS_ATP_TST_FLOW_HMACDYNAMIC_AddBuffers_Acceptance();
    
    TST_ATP_FLOW_Error = TST_ATP_FLOW_Error+TST_ATP_Error;
    
    #if CRYS_ATP_TST_EXIT_SYS_TST_ON_ERROR
    if (TST_ATP_FLOW_Error != TESTPASS)
	    return 	TST_ATP_FLOW_Error;
    #endif
    
        
   return TST_ATP_FLOW_Error;
}


#endif
